package org.msh.tb.reports2.variables;

import org.apache.commons.lang.StringUtils;
import org.jboss.seam.core.Expressions;
import org.jboss.seam.international.Messages;
import org.msh.reports.filters.FilterOption;
import org.msh.reports.filters.ValueHandler;
import org.msh.reports.keys.Key;
import org.msh.tb.entities.enums.MessageKey;
import org.msh.tb.reports2.VariableImpl;

import java.util.ArrayList;
import java.util.List;


/**
 * Variable that handles fields of type enumeration
 *
 * @author Ricardo Memoria
 *
 */
public class EnumFieldVariable extends VariableImpl {

    private Class<? extends Enum> enumClass;
    private Enum[] options;
    private String optionsExpression;

    public EnumFieldVariable(String id, String keylabel, String fieldName, Class<? extends Enum> enumClass) {
        super(id, keylabel, fieldName);
        this.enumClass = enumClass;
    }


    public EnumFieldVariable(String id, String keylabel, String fieldName, Class<? extends Enum> enumClass, Enum[] options) {
        super(id, keylabel, fieldName);
        this.enumClass = enumClass;
        this.options = options;
    }


    public EnumFieldVariable(String id, String keylabel, String fieldName, Class<? extends Enum> enumClass, String optionsExpression) {
        super(id, keylabel, fieldName);
        this.enumClass = enumClass;
        this.optionsExpression = optionsExpression;
    }


    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#getDisplayText(java.lang.Object)
     */
    @Override
    public String getDisplayText(Key key) {
        if (key.isNull()) {
            return Messages.instance().get("global.notdef");
        }

        Enum val = null;
        Enum[] options = getEnumClass().getEnumConstants();

        if (key.getValue() instanceof Number) {
            int index = ((Number)key.getValue()).intValue();
            val = options[index];
        } else if (key.getValue() instanceof Enum) {
            val = (Enum)key.getValue();
        }

        if (val == null) {
            return key.getValue().toString();
        }

        String msgkey = val.getClass().getSimpleName() + "." + val.toString();
        return Messages.instance().get(msgkey);
    }

    @Override
    protected ValueHandler.ValueConverter getConverter() {
        return new ValueHandler.ValueConverter() {
            @Override
            public Object convert(String value) {
                if (StringUtils.isNumeric(value)) {
                    return value;
                }
                // search for value in list of enums
                Enum[] vals = getEnumValues();
                for (Enum e: vals) {
                    if (e.toString().equals(value)) {
                        return e.ordinal();
                    }
                }
                return null;
            }
        };
    }

    /* (non-Javadoc)
         * @see org.msh.tb.reports2.VariableImpl#createKey(java.lang.Object)
         */
    @Override
    public Key createKey(Object value, int iteration) {
        if (value == null) {
            return Key.asNull();
        }

        if (value instanceof Number) {
            return Key.of(((Number)value).intValue());
        }

        return super.createKey(value, iteration);
    }

    /**
     * Return the list of enumeration values available
     * @return
     */
    protected Enum[] getEnumValues() {
        // list of options was given ?
        if (options != null) {
            return options;
        }

        // an expression to return the options was given ?
        if (optionsExpression != null) {
            return (Enum[])Expressions.instance().createValueExpression(optionsExpression).getValue();
        }

        if (enumClass == null) {
            return null;
        }

        return enumClass.getEnumConstants();
    }

    /**
     * @return the enumClass
     */
    public Class<? extends Enum> getEnumClass() {
        return enumClass;
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#getFilterOptions()
     */
    @Override
    public List<FilterOption> getFilterOptions(Object param) {
        Enum[] vals = getEnumValues();

        List<FilterOption> lst = new ArrayList<FilterOption>();
        for (Enum val: vals) {
            String key;
            // get the display name of the label
            if (val instanceof MessageKey) {
                key = ((MessageKey)val).getMessageKey();
            } else {
                key = val.getClass().getSimpleName() + "." + val.toString();
            }
            String label = Messages.instance().get(key);
            // add in the list of options
            lst.add(new FilterOption(val.ordinal(), label));
        }
        return lst;
    }


    /**
     * @return the options
     */
    public Enum[] getOptions() {
        return options;
    }


    /**
     * @param options the options to set
     */
    public void setOptions(Enum[] options) {
        this.options = options;
    }


    /**
     * @return the optionsExpression
     */
    public String getOptionsExpression() {
        return optionsExpression;
    }


    /**
     * @param optionsExpression the optionsExpression to set
     */
    public void setOptionsExpression(String optionsExpression) {
        this.optionsExpression = optionsExpression;
    }


    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#compareValues(java.lang.Object, java.lang.Object)
     */
    @Override
    public int compareValues(Key val1, Key val2) {
        return val1.compareTo(val2);
    }


    /** {@inheritDoc}
     */
    @Override
    public Object[] getDomain() {
        return super.getDomain();
    }

    @Override
    public boolean isMultiSelection() {
        Enum[] options = getEnumValues();
        return (options != null && options.length > 2);
    }
}
